/*++

INTEL CONFIDENTIAL
Copyright 2013-2018 Intel Corporation All Rights Reserved.

The source code contained or described herein and all documents
related to the source code ("Material") are owned by Intel Corporation
or its suppliers or licensors. Title to the Material remains with
Intel Corporation or its suppliers and licensors. The Material
contains trade secrets and proprietary and confidential information of
Intel or its suppliers and licensors. The Material is protected by
worldwide copyright and trade secret laws and treaty provisions. No
part of the Material may be used, copied, reproduced, modified,
published, uploaded, posted, transmitted, distributed, or disclosed in
any way without Intel's prior express written permission.

No license under any patent, copyright, trade secret or other
intellectual property right is granted to or conferred upon you by
disclosure or delivery of the Materials, either expressly, by
implication, inducement, estoppel or otherwise. Any license under such
intellectual property rights must be express and approved by Intel in
writing.

File Name:
icc_sdk_api.h

Abstract:
This is the interface of the API of icc sdk package.

This file is needed for linking the "icc_sdk" binaries, to your project.

Usage:
1. add this file to your project
2. add "IccLibStatic.lib" binary file, to your project
3. add "COMPILE_WITH_ICC_SDK_BINARIES" definition, to your project
4. compile
--*/

#ifndef _ICC_SDK_API_H
#define _ICC_SDK_API_H

#ifdef DLL_BUILD
#ifndef _DLL_EXPORT_OPTION_
#define _DLL_EXPORT_OPTION_
#endif
#define EXTERNAL_API __declspec(dllexport)
#else
#define EXTERNAL_API
#endif

#ifndef COMPILE_WITH_ICC_SDK_BINARIES
#include "types.h"
#include <icc_heci_types.h>
#include <icc_api.h>
#endif

#ifdef COMPILE_WITH_ICC_SDK_BINARIES
/*
* These definitions are originally defined
* (and should be the same as)
* in "icc_heci_types.h"
*
* Definition below is for linking this header, with compiled icc_sdk binaries,
* see documentation above
*/

/*
* Clock ID's that are used to communicate in external interfaces (like: heci),
* Numbers should NEVER be changed!!!
* New IDs can be added at the end of enum only.
* Use macro: EXTERNAL_CLOCK_ID_TO_INTERNAL_CLOCK_ID, to convert external clock-id, to internal clock-id
*/

typedef enum
{
    ICC_HECI_PCIE_CLOCK_ID = 0, // CPUPCIEBCLK Signal to CPU
    ICC_HECI_BCLK_CLOCK_ID = 1, //CPUBCLK Signal to CPU
    ICC_HECI_WMPHY_CLOCK_ID = 2 //WMPHY PLL Signal (SPS ONLY!)
}ICC_HECI_CLOCK_ID;

typedef union _GET_MPHY_VERSION
{
    UINT32 data;
    struct
    {
        UINT32 CRC : 16;
        UINT32 Ver : 8;
        UINT32 Product_and_Stepping : 8;
    } Fields;
}GET_MPHY_VERSION;

#endif


typedef struct _ICC_SET_CLOCK_SETTINGSEx
{
    UINT32 UserFrequency;
    UINT8  SscPercent;   // encoding example: 1.28% -> SSC_SPREAD value is 128
    BOOL   SetToDefault;
    BOOL   ForcePowerFlow;
} ICC_SET_CLOCK_SETTINGSEx;

typedef struct _ICC_GET_CLOCK_SETTINGSEx
{
    UINT32 Frequency;
    UINT32 UserFrequency;
    UINT32 MaxFrequency;
    UINT32 MinFrequency;
    UINT8  SscMode;
    UINT8  SscPercent; // encoding example: 1.28% -> SSC_SPREAD value is 128
    UINT8  MaxSscPercent;
    //    ICC_DMI_PEG_RATIO DmiPegRatio; type, set to default, warm reset
    UINT16 CurrentFlags;       //Bits field not supported in proxy compilation
    UINT16 SupportFlags;
} ICC_GET_CLOCK_SETTINGSEx;
#ifdef __cplusplus
extern "C" {
#endif

/**
********************************************************************************
* IccLibGetCurrentClockSettingsWrapper
*   Returns current settings of given clock.
*
* @param[in]  clockId        Selects clock
* @param[out] clockSettings  Current settings of the selected clock
*
* @return status
*
*********************************************************************************/
EXTERNAL_API UINT32 IccLibGetCurrentClockSettingsWrapper(const ICC_HECI_CLOCK_ID clockId,
                                                         ICC_GET_CLOCK_SETTINGSEx * const clockSettings);

/**
********************************************************************************
* IccLibSetCurrentClockSettingsWrapper
*  Immediately changes settings of a given clock to the requested value,
*  leaving all others intact. Error will be returned on range violation.
*
* @param[in] clockId       Id of clock settings to set
* @param[in, out] clockSettings New clock settings to program
*                 In this struct, the only 'out' field for now is 'frequency'
*
* @return status
*
*********************************************************************************/
EXTERNAL_API UINT32 IccLibSetCurrentClockSettingsWrapper(const ICC_HECI_CLOCK_ID clockId,
                                                         ICC_SET_CLOCK_SETTINGSEx * clockSettings);
 
/**
********************************************************************************
* IccLibSetMphySettingsWrapper
*  Changes MPHY table current settings to the requested value,
*  leaving all others intact. Error will be returned on range violation.
*
* @param[in] mphyFileName       the name of the settings file

* @return status
*
*********************************************************************************/
EXTERNAL_API UINT32 IccLibSetMphySettingsWrapper(char *mphyFileName);

/**
********************************************************************************
* IccLibGetMphySettingsWrapper
*  Returns MPHY table current settings
*
* @param[in] length       num of bytes to read
* @param[in] offset       start offset
* @param[out] buffer      current settings of mphy table
* @param[out] bytesRead   num of bytes read
*
* @return status
*
*********************************************************************************/
EXTERNAL_API UINT32 IccLibGetMphySettingsWrapper(UINT32 length, UINT32 offset, UINT8 *buffer,UINT32 *bytesRead);

/**
********************************************************************************
* IccLibGetMphyVersion
*  Returns MPHY table version
*

* @param[out] survTable     MPHY table version
*
* @return status
*
*********************************************************************************/
EXTERNAL_API UINT32 IccLibGetMphyVersion(GET_MPHY_VERSION *survTable);

/**
* @brief Print error message according to error code.
*
* @param[in]  errorCode  Error code.
*/
EXTERNAL_API const char* GetErrorStringByCode(const UINT32 errorCode);

#ifdef __cplusplus
}
#endif
#endif
